// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { NodeDefaultCryptographicMaterialsManager, NodeAlgorithmSuite, KeyringNode, getEncryptHelper, needs, CommitmentPolicy, CommitmentPolicySuites, MessageFormat, } from '@aws-crypto/material-management-node';
import { getFramedEncryptStream } from './framed_encrypt_stream';
import { SignatureStream } from './signature_stream';
import Duplexify from 'duplexify';
import { randomBytes } from 'crypto';
import { serializeFactory, FRAME_LENGTH, Maximum, MessageIdLength, serializeMessageHeaderAuth, } from '@aws-crypto/serialize';
// @ts-ignore
import { pipeline } from 'readable-stream';
const fromUtf8 = (input) => Buffer.from(input, 'utf8');
const { serializeMessageHeader, headerAuthIv, buildMessageHeader } = serializeFactory(fromUtf8);
/**
 * Takes a NodeDefaultCryptographicMaterialsManager or a KeyringNode that will
 * be wrapped in a NodeDefaultCryptographicMaterialsManager and returns a stream.
 *
 * @param commitmentPolicy
 * @param maxEncryptedDataKeys
 * @param cmm NodeMaterialsManager|KeyringNode
 * @param op EncryptStreamInput
 */
export function _encryptStream({ commitmentPolicy, maxEncryptedDataKeys }, cmm, op = {}) {
    /* Precondition: encryptStream needs a valid commitmentPolicy. */
    needs(CommitmentPolicy[commitmentPolicy], 'Invalid commitment policy.');
    // buildEncrypt defaults this to false for backwards compatibility, so this is satisfied
    /* Precondition: encryptStream needs a valid maxEncryptedDataKeys. */
    needs(maxEncryptedDataKeys === false || maxEncryptedDataKeys >= 1, 'Invalid maxEncryptedDataKeys value.');
    const { suiteId, encryptionContext = {}, frameLength = FRAME_LENGTH, plaintextLength, } = op;
    /* Precondition: The frameLength must be less than the maximum frame size Node.js stream. */
    needs(frameLength > 0 && Maximum.FRAME_SIZE >= frameLength, `frameLength out of bounds: 0 > frameLength >= ${Maximum.FRAME_SIZE}`);
    /* If the cmm is a Keyring, wrap it with NodeDefaultCryptographicMaterialsManager. */
    cmm =
        cmm instanceof KeyringNode
            ? new NodeDefaultCryptographicMaterialsManager(cmm)
            : cmm;
    const suite = suiteId && new NodeAlgorithmSuite(suiteId);
    /* Precondition: Only request NodeEncryptionMaterial for algorithm suites supported in commitmentPolicy. */
    CommitmentPolicySuites.isEncryptEnabled(commitmentPolicy, suite);
    const wrappingStream = new Duplexify();
    cmm
        .getEncryptionMaterials({
        suite,
        encryptionContext,
        plaintextLength,
        commitmentPolicy,
    })
        .then(async (material) => {
        /* Precondition: Only use NodeEncryptionMaterial for algorithm suites supported in commitmentPolicy. */
        CommitmentPolicySuites.isEncryptEnabled(commitmentPolicy, material.suite);
        /* Precondition: _encryptStream encryption materials must not exceed maxEncryptedDataKeys */
        needs(maxEncryptedDataKeys === false ||
            material.encryptedDataKeys.length <= maxEncryptedDataKeys, 'maxEncryptedDataKeys exceeded.');
        const { getCipher, messageHeader, rawHeader, dispose, getSigner } = getEncryptionInfo(material, frameLength);
        wrappingStream.emit('MessageHeader', messageHeader);
        const encryptStream = getFramedEncryptStream(getCipher, messageHeader, dispose, { plaintextLength, suite: material.suite });
        const signatureStream = new SignatureStream(getSigner);
        pipeline(encryptStream, signatureStream);
        wrappingStream.setReadable(signatureStream);
        // Flush the rawHeader through the signatureStream
        rawHeader.forEach((buff) => signatureStream.write(buff));
        // @ts-ignore until readable-stream exports v3 types...
        wrappingStream.setWritable(encryptStream);
    })
        .catch((err) => wrappingStream.emit('error', err));
    return wrappingStream;
}
export function getEncryptionInfo(material, frameLength) {
    const { getCipherInfo, dispose, getSigner } = getEncryptHelper(material);
    const { suite, encryptionContext, encryptedDataKeys } = material;
    const { ivLength, messageFormat } = material.suite;
    const versionString = MessageFormat[messageFormat];
    const messageIdLength = parseInt(MessageIdLength[versionString], 10);
    /* Precondition UNTESTED: Node suites must result is some messageIdLength. */
    needs(messageIdLength > 0, 'Algorithm suite has unknown message format.');
    const messageId = randomBytes(messageIdLength);
    const { getCipher, keyCommitment } = getCipherInfo(messageId);
    const messageHeader = buildMessageHeader({
        suite: suite,
        encryptedDataKeys,
        encryptionContext,
        messageId,
        frameLength,
        suiteData: keyCommitment,
    });
    const { buffer, byteOffset, byteLength } = serializeMessageHeader(messageHeader);
    const headerBuffer = Buffer.from(buffer, byteOffset, byteLength);
    const headerIv = headerAuthIv(ivLength);
    const validateHeader = getCipher(headerIv);
    validateHeader.setAAD(headerBuffer);
    validateHeader.update(Buffer.alloc(0));
    validateHeader.final();
    const headerAuthTag = validateHeader.getAuthTag();
    return {
        getCipher,
        dispose,
        getSigner,
        messageHeader,
        rawHeader: [
            headerBuffer,
            serializeMessageHeaderAuth({ headerIv, headerAuthTag, messageHeader }),
        ],
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZW5jcnlwdF9zdHJlYW0uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvZW5jcnlwdF9zdHJlYW0udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUEsb0VBQW9FO0FBQ3BFLHNDQUFzQztBQUV0QyxPQUFPLEVBQ0wsd0NBQXdDLEVBQ3hDLGtCQUFrQixFQUVsQixXQUFXLEVBRVgsZ0JBQWdCLEVBR2hCLEtBQUssRUFDTCxnQkFBZ0IsRUFDaEIsc0JBQXNCLEVBQ3RCLGFBQWEsR0FFZCxNQUFNLHNDQUFzQyxDQUFBO0FBQzdDLE9BQU8sRUFBRSxzQkFBc0IsRUFBRSxNQUFNLHlCQUF5QixDQUFBO0FBQ2hFLE9BQU8sRUFBRSxlQUFlLEVBQUUsTUFBTSxvQkFBb0IsQ0FBQTtBQUNwRCxPQUFPLFNBQVMsTUFBTSxXQUFXLENBQUE7QUFDakMsT0FBTyxFQUFFLFdBQVcsRUFBRSxNQUFNLFFBQVEsQ0FBQTtBQUNwQyxPQUFPLEVBQ0wsZ0JBQWdCLEVBQ2hCLFlBQVksRUFDWixPQUFPLEVBQ1AsZUFBZSxFQUNmLDBCQUEwQixHQUMzQixNQUFNLHVCQUF1QixDQUFBO0FBRTlCLGFBQWE7QUFDYixPQUFPLEVBQUUsUUFBUSxFQUFFLE1BQU0saUJBQWlCLENBQUE7QUFHMUMsTUFBTSxRQUFRLEdBQUcsQ0FBQyxLQUFhLEVBQUUsRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFBO0FBQzlELE1BQU0sRUFBRSxzQkFBc0IsRUFBRSxZQUFZLEVBQUUsa0JBQWtCLEVBQUUsR0FDaEUsZ0JBQWdCLENBQUMsUUFBUSxDQUFDLENBQUE7QUFTNUI7Ozs7Ozs7O0dBUUc7QUFDSCxNQUFNLFVBQVUsY0FBYyxDQUM1QixFQUFFLGdCQUFnQixFQUFFLG9CQUFvQixFQUFpQixFQUN6RCxHQUF1QyxFQUN2QyxLQUF5QixFQUFFO0lBRTNCLGlFQUFpRTtJQUNqRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsZ0JBQWdCLENBQUMsRUFBRSw0QkFBNEIsQ0FBQyxDQUFBO0lBRXZFLHdGQUF3RjtJQUN4RixxRUFBcUU7SUFDckUsS0FBSyxDQUNILG9CQUFvQixLQUFLLEtBQUssSUFBSSxvQkFBb0IsSUFBSSxDQUFDLEVBQzNELHFDQUFxQyxDQUN0QyxDQUFBO0lBRUQsTUFBTSxFQUNKLE9BQU8sRUFDUCxpQkFBaUIsR0FBRyxFQUFFLEVBQ3RCLFdBQVcsR0FBRyxZQUFZLEVBQzFCLGVBQWUsR0FDaEIsR0FBRyxFQUFFLENBQUE7SUFFTiw0RkFBNEY7SUFDNUYsS0FBSyxDQUNILFdBQVcsR0FBRyxDQUFDLElBQUksT0FBTyxDQUFDLFVBQVUsSUFBSSxXQUFXLEVBQ3BELGlEQUFpRCxPQUFPLENBQUMsVUFBVSxFQUFFLENBQ3RFLENBQUE7SUFFRCxxRkFBcUY7SUFDckYsR0FBRztRQUNELEdBQUcsWUFBWSxXQUFXO1lBQ3hCLENBQUMsQ0FBQyxJQUFJLHdDQUF3QyxDQUFDLEdBQUcsQ0FBQztZQUNuRCxDQUFDLENBQUMsR0FBRyxDQUFBO0lBRVQsTUFBTSxLQUFLLEdBQUcsT0FBTyxJQUFJLElBQUksa0JBQWtCLENBQUMsT0FBTyxDQUFDLENBQUE7SUFFeEQsMkdBQTJHO0lBQzNHLHNCQUFzQixDQUFDLGdCQUFnQixDQUFDLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxDQUFBO0lBRWhFLE1BQU0sY0FBYyxHQUFHLElBQUksU0FBUyxFQUFFLENBQUE7SUFFdEMsR0FBRztTQUNBLHNCQUFzQixDQUFDO1FBQ3RCLEtBQUs7UUFDTCxpQkFBaUI7UUFDakIsZUFBZTtRQUNmLGdCQUFnQjtLQUNqQixDQUFDO1NBQ0QsSUFBSSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsRUFBRTtRQUN2Qix1R0FBdUc7UUFDdkcsc0JBQXNCLENBQUMsZ0JBQWdCLENBQUMsZ0JBQWdCLEVBQUUsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFBO1FBRXpFLDRGQUE0RjtRQUM1RixLQUFLLENBQ0gsb0JBQW9CLEtBQUssS0FBSztZQUM1QixRQUFRLENBQUMsaUJBQWlCLENBQUMsTUFBTSxJQUFJLG9CQUFvQixFQUMzRCxnQ0FBZ0MsQ0FDakMsQ0FBQTtRQUVELE1BQU0sRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsU0FBUyxFQUFFLEdBQy9ELGlCQUFpQixDQUFDLFFBQVEsRUFBRSxXQUFXLENBQUMsQ0FBQTtRQUUxQyxjQUFjLENBQUMsSUFBSSxDQUFDLGVBQWUsRUFBRSxhQUFhLENBQUMsQ0FBQTtRQUVuRCxNQUFNLGFBQWEsR0FBRyxzQkFBc0IsQ0FDMUMsU0FBUyxFQUNULGFBQWEsRUFDYixPQUFPLEVBQ1AsRUFBRSxlQUFlLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxLQUFLLEVBQUUsQ0FDM0MsQ0FBQTtRQUNELE1BQU0sZUFBZSxHQUFHLElBQUksZUFBZSxDQUFDLFNBQVMsQ0FBQyxDQUFBO1FBRXRELFFBQVEsQ0FBQyxhQUFhLEVBQUUsZUFBZSxDQUFDLENBQUE7UUFFeEMsY0FBYyxDQUFDLFdBQVcsQ0FBQyxlQUFlLENBQUMsQ0FBQTtRQUMzQyxrREFBa0Q7UUFDbEQsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFBO1FBRXhELHVEQUF1RDtRQUN2RCxjQUFjLENBQUMsV0FBVyxDQUFDLGFBQWEsQ0FBQyxDQUFBO0lBQzNDLENBQUMsQ0FBQztTQUNELEtBQUssQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQTtJQUVwRCxPQUFPLGNBQWMsQ0FBQTtBQUN2QixDQUFDO0FBRUQsTUFBTSxVQUFVLGlCQUFpQixDQUMvQixRQUFnQyxFQUNoQyxXQUFtQjtJQUVuQixNQUFNLEVBQUUsYUFBYSxFQUFFLE9BQU8sRUFBRSxTQUFTLEVBQUUsR0FBRyxnQkFBZ0IsQ0FBQyxRQUFRLENBQUMsQ0FBQTtJQUN4RSxNQUFNLEVBQUUsS0FBSyxFQUFFLGlCQUFpQixFQUFFLGlCQUFpQixFQUFFLEdBQUcsUUFBUSxDQUFBO0lBQ2hFLE1BQU0sRUFBRSxRQUFRLEVBQUUsYUFBYSxFQUFFLEdBQUcsUUFBUSxDQUFDLEtBQUssQ0FBQTtJQUVsRCxNQUFNLGFBQWEsR0FBRyxhQUFhLENBQUMsYUFBYSxDQUFRLENBQUE7SUFDekQsTUFBTSxlQUFlLEdBQUcsUUFBUSxDQUFDLGVBQWUsQ0FBQyxhQUFhLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQTtJQUNwRSw2RUFBNkU7SUFDN0UsS0FBSyxDQUFDLGVBQWUsR0FBRyxDQUFDLEVBQUUsNkNBQTZDLENBQUMsQ0FBQTtJQUN6RSxNQUFNLFNBQVMsR0FBRyxXQUFXLENBQUMsZUFBZSxDQUFDLENBQUE7SUFFOUMsTUFBTSxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsR0FBRyxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUE7SUFFN0QsTUFBTSxhQUFhLEdBQUcsa0JBQWtCLENBQUM7UUFDdkMsS0FBSyxFQUFFLEtBQUs7UUFDWixpQkFBaUI7UUFDakIsaUJBQWlCO1FBQ2pCLFNBQVM7UUFDVCxXQUFXO1FBQ1gsU0FBUyxFQUFFLGFBQWE7S0FDekIsQ0FBQyxDQUFBO0lBRUYsTUFBTSxFQUFFLE1BQU0sRUFBRSxVQUFVLEVBQUUsVUFBVSxFQUFFLEdBQ3RDLHNCQUFzQixDQUFDLGFBQWEsQ0FBQyxDQUFBO0lBQ3ZDLE1BQU0sWUFBWSxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLFVBQVUsRUFBRSxVQUFVLENBQUMsQ0FBQTtJQUNoRSxNQUFNLFFBQVEsR0FBRyxZQUFZLENBQUMsUUFBUSxDQUFDLENBQUE7SUFDdkMsTUFBTSxjQUFjLEdBQUcsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFBO0lBQzFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsWUFBWSxDQUFDLENBQUE7SUFDbkMsY0FBYyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUE7SUFDdEMsY0FBYyxDQUFDLEtBQUssRUFBRSxDQUFBO0lBQ3RCLE1BQU0sYUFBYSxHQUFHLGNBQWMsQ0FBQyxVQUFVLEVBQUUsQ0FBQTtJQUVqRCxPQUFPO1FBQ0wsU0FBUztRQUNULE9BQU87UUFDUCxTQUFTO1FBQ1QsYUFBYTtRQUNiLFNBQVMsRUFBRTtZQUNULFlBQVk7WUFDWiwwQkFBMEIsQ0FBQyxFQUFFLFFBQVEsRUFBRSxhQUFhLEVBQUUsYUFBYSxFQUFFLENBQUM7U0FDdkU7S0FDRixDQUFBO0FBQ0gsQ0FBQyJ9