﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/lexv2-models/LexModelsV2Request.h>
#include <aws/lexv2-models/LexModelsV2_EXPORTS.h>
#include <aws/lexv2-models/model/EncryptionSetting.h>
#include <aws/lexv2-models/model/TranscriptSourceSetting.h>

#include <utility>

namespace Aws {
namespace LexModelsV2 {
namespace Model {

/**
 */
class StartBotRecommendationRequest : public LexModelsV2Request {
 public:
  AWS_LEXMODELSV2_API StartBotRecommendationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartBotRecommendation"; }

  AWS_LEXMODELSV2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The unique identifier of the bot containing the bot recommendation.</p>
   */
  inline const Aws::String& GetBotId() const { return m_botId; }
  inline bool BotIdHasBeenSet() const { return m_botIdHasBeenSet; }
  template <typename BotIdT = Aws::String>
  void SetBotId(BotIdT&& value) {
    m_botIdHasBeenSet = true;
    m_botId = std::forward<BotIdT>(value);
  }
  template <typename BotIdT = Aws::String>
  StartBotRecommendationRequest& WithBotId(BotIdT&& value) {
    SetBotId(std::forward<BotIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the bot containing the bot recommendation.</p>
   */
  inline const Aws::String& GetBotVersion() const { return m_botVersion; }
  inline bool BotVersionHasBeenSet() const { return m_botVersionHasBeenSet; }
  template <typename BotVersionT = Aws::String>
  void SetBotVersion(BotVersionT&& value) {
    m_botVersionHasBeenSet = true;
    m_botVersion = std::forward<BotVersionT>(value);
  }
  template <typename BotVersionT = Aws::String>
  StartBotRecommendationRequest& WithBotVersion(BotVersionT&& value) {
    SetBotVersion(std::forward<BotVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the language and locale of the bot recommendation to start.
   * The string must match one of the supported locales. For more information, see <a
   * href="https://docs.aws.amazon.com/lexv2/latest/dg/how-languages.html">Supported
   * languages</a> </p>
   */
  inline const Aws::String& GetLocaleId() const { return m_localeId; }
  inline bool LocaleIdHasBeenSet() const { return m_localeIdHasBeenSet; }
  template <typename LocaleIdT = Aws::String>
  void SetLocaleId(LocaleIdT&& value) {
    m_localeIdHasBeenSet = true;
    m_localeId = std::forward<LocaleIdT>(value);
  }
  template <typename LocaleIdT = Aws::String>
  StartBotRecommendationRequest& WithLocaleId(LocaleIdT&& value) {
    SetLocaleId(std::forward<LocaleIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The object representing the Amazon S3 bucket containing the transcript, as
   * well as the associated metadata.</p>
   */
  inline const TranscriptSourceSetting& GetTranscriptSourceSetting() const { return m_transcriptSourceSetting; }
  inline bool TranscriptSourceSettingHasBeenSet() const { return m_transcriptSourceSettingHasBeenSet; }
  template <typename TranscriptSourceSettingT = TranscriptSourceSetting>
  void SetTranscriptSourceSetting(TranscriptSourceSettingT&& value) {
    m_transcriptSourceSettingHasBeenSet = true;
    m_transcriptSourceSetting = std::forward<TranscriptSourceSettingT>(value);
  }
  template <typename TranscriptSourceSettingT = TranscriptSourceSetting>
  StartBotRecommendationRequest& WithTranscriptSourceSetting(TranscriptSourceSettingT&& value) {
    SetTranscriptSourceSetting(std::forward<TranscriptSourceSettingT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The object representing the passwords that will be used to encrypt the data
   * related to the bot recommendation results, as well as the KMS key ARN used to
   * encrypt the associated metadata.</p>
   */
  inline const EncryptionSetting& GetEncryptionSetting() const { return m_encryptionSetting; }
  inline bool EncryptionSettingHasBeenSet() const { return m_encryptionSettingHasBeenSet; }
  template <typename EncryptionSettingT = EncryptionSetting>
  void SetEncryptionSetting(EncryptionSettingT&& value) {
    m_encryptionSettingHasBeenSet = true;
    m_encryptionSetting = std::forward<EncryptionSettingT>(value);
  }
  template <typename EncryptionSettingT = EncryptionSetting>
  StartBotRecommendationRequest& WithEncryptionSetting(EncryptionSettingT&& value) {
    SetEncryptionSetting(std::forward<EncryptionSettingT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_botId;
  bool m_botIdHasBeenSet = false;

  Aws::String m_botVersion;
  bool m_botVersionHasBeenSet = false;

  Aws::String m_localeId;
  bool m_localeIdHasBeenSet = false;

  TranscriptSourceSetting m_transcriptSourceSetting;
  bool m_transcriptSourceSettingHasBeenSet = false;

  EncryptionSetting m_encryptionSetting;
  bool m_encryptionSettingHasBeenSet = false;
};

}  // namespace Model
}  // namespace LexModelsV2
}  // namespace Aws
